<?php

/**
 * Core: The HTTP Request class
 *
 * @link https://oxyprops.com
 *
 * @package    OxyProps
 * @subpackage Core
 * @author     Cédric Bontems <cedric@thewebforge.dev>
 * @since      1.5.2
 * @copyright  Copyright (c) 2022, Cédric Bontems
 * @license    https://www.gnu.org/licenses/gpl-2.0.html  GPLv2 or later
 */

namespace OxyProps\Inc;

// If this file is called directly, abort.
if (!defined('ABSPATH')) {
	exit;
}

/**
 * OxyProps\Inc\Oxyprops class.
 *
 * Core class used to organize plugin related data.
 *
 * @since  1.5.0 Creation.
 * @author Cédric Bontems <cedric@thewebforge.dev>
 */
class HTTPRequest
{
	var $_fp;        // HTTP socket
	var $_url;        // full URL
	var $_host;        // HTTP host
	var $_protocol;    // protocol (HTTP/HTTPS)
	var $_uri;        // request URI
	var $_port;        // port

	// scan url

	public function _scan_url()
	{
		$req = $this->_url;
		$pos = strpos($req, '://');
		$this->_protocol = strtolower(substr($req, 0, $pos));
		$req = substr($req, $pos + 3);
		$pos = strpos($req, '/');
		if ($pos === false) {
			$pos = strlen($req);
		}
		$host = substr($req, 0, $pos);
		if (strpos($host, ':') !== false) {
			list($this->_host, $this->_port) = explode(':', $host);
		} else {
			$this->_host = $host;
			$this->_port = ($this->_protocol == 'https') ? 443 : 80;
		}
		$this->_uri = substr($req, $pos);
		if ($this->_uri == '') {
			$this->_uri = '/';
		}
	}

	// constructor

	public function __construct($url)
	{
		$this->_url = $url;
		$this->_scan_url();
	}

	public function error_handler($errno, $errstr, $errfile, $errline)
	{
		if (($errno & error_reporting()) > 0)
			throw new \ErrorException($errstr, 500, $errno, $errfile, $errline);
		else
			return false;
	}

	// download URL to string

	public function DownloadToString()
	{
		$crlf = "\r\n";

		// generate request

		$req = 'GET ' . $this->_uri . ' HTTP/1.0' . $crlf
			.    'Host: ' . $this->_host . $crlf
			.    $crlf;

		// fetch
		set_error_handler(array($this, 'error_handler'));
		try {
			$this->_fp = fsockopen(($this->_protocol == 'https' ? 'ssl://' : '') . $this->_host, $this->_port);
		} catch (\Exception $ex) { //used back-slash for global namespace
			return '';
		}

		$response = '';
		if (!is_resource($this->_fp)) return '';
		fwrite($this->_fp, $req);
		while (is_resource($this->_fp) && $this->_fp && !feof($this->_fp)) {
			$response .= fread($this->_fp, 1024);
		}
		fclose($this->_fp);

		// split header and body
		$pos = strpos($response, $crlf . $crlf);
		if ($pos === false) {
			return ($response);
		}
		$header = substr($response, 0, $pos);
		$body = substr($response, $pos + 2 * strlen($crlf));

		// parse headers
		$headers = array();
		$lines = explode($crlf, $header);
		foreach ($lines as $line) {
			if (($pos = strpos($line, ':')) !== false) {
				$headers[strtolower(trim(substr($line, 0, $pos)))] = trim(substr($line, $pos + 1));
			}
		}

		// redirection?
		if (isset($headers['location'])) {
			$http = new HTTPRequest($headers['location']);
			return ($http->DownloadToString($http));
		} else {
			return ($body);
		}
	}
}
